"""check_egiis - EGIIS infosystem checks."""

from arcnagios import nagutils
from arcnagios.infosys import arcinfosys, ldaputils
from arcnagios.infosys.ldaputils import ldap_entry_dn
from arcnagios.infosys.ldap_nagios_plugin import LDAPNagiosPlugin
from arcnagios.infosys.arcinfosys import MdsServiceLdap
from arcnagios.confargparse import UsageError

class Check_egiis(LDAPNagiosPlugin):

    main_config_section = 'arcinfosys'

    def __init__(self):
        LDAPNagiosPlugin.__init__(self, use_port = True, default_port = 2135)
        ap = self.argparser.add_argument_group('EGIIS Options')
        ap.add_argument('--index-name', dest = 'index_name',
                help = 'The name of the information index to query.')

    def parse_args(self, args):
        LDAPNagiosPlugin.parse_args(self, args)
        if not self.opts.ldap_basedn and not self.opts.index_name:
            raise UsageError('--ldap-basedn or --index-name must be specified.')

    def valid_egiis_entry(self, ent):
        """Validate a single EGIIS entry and return an pair of the number of
        errors and an `EGIIS_Object` for the entry or None."""

        try:
            return MdsServiceLdap(self._ldap_server, ent)
        except ldaputils.LDAPValidationError as exn:
            self.log.error(str(exn))
            self.nagios_report.update_status_code(nagutils.CRITICAL)
            return None

    def check(self):
        self.prepare_check()

        # Query the EGIIS.
        basedn = self.opts.ldap_basedn \
                or 'Mds-Vo-name=%s,o=grid'%self.opts.index_name
        sr = self.ldap_search(basedn, '(objectClass=MdsServiceLdap)',
                search_scope = 'BASE')
        if sr == []:
            raise nagutils.ServiceWarning('No EGIIS entries found.')

        # Check the entries.
        entcnts = {}
        for ent in sr:
            egiis = self.valid_egiis_entry(ent)
            if egiis:
                self.log.info('Good entry %s', ldap_entry_dn(ent))
                entcnts[egiis.reg_status] = 1 + entcnts.get(egiis.reg_status, 0)

        # Report the result.
        entcnts = list(entcnts.items())
        entcnts.sort(key = lambda x: x[0])
        countstrings = \
            ['%d %s'%(cnt, arcinfosys.egiis_reg_status_to_string(st).lower())
             for st, cnt in entcnts]
        self.nagios_report.update_status(
                nagutils.OK, 'EGIIS ok: %s.'%', '.join(countstrings))
        return self.nagios_exit(subject = 'EGIIS service')
