package Sisimai::Message::JSON;
use parent 'Sisimai::Message';
use feature ':5.10';
use strict;
use warnings;

sub make {
    # Make data structure from decoded JSON object
    # @param         [Hash] argvs   Bounce object
    # @options argvs [Hash]  data   Decoded JSON
    # @options argvs [Array] load   User defined MTA module list
    # @options argvs [Array] order  The order of MTA modules
    # @options argvs [Code]  hook   Reference to callback method
    # @return        [Hash]         Resolved data structure
    my $class = shift; $class->SUPER::warn('');
    return $class->SUPER::make({ $_ });
}

1;
__END__

=encoding utf-8

=head1 NAME

Sisimai::Message::JSON - Convert bounce object (decoded JSON) to data structure.

=head1 SYNOPSIS

    use JSON;
    use Sisimai::Message;

    my $jsonparser = JSON->new;
    my $jsonobject = $jsonparser->decode($jsonstring);
    my $messageobj = Sisimai::Message->new('data' => $jsonobject, 'input' => 'json');

=head1 DESCRIPTION

Sisimai::Message::JSON convert bounce object (decode JSON) to data structure.
When the email given as a argument of "new()" method is not a decoded JSON, the
method returns "undef".

=head1 CLASS METHODS

=head2 C<B<new(I<Hash reference>)>>

C<new()> is a constructor of Sisimai::Message

    my $jsonparser = JSON->new;
    my $jsonstring = '{"neko":2, "nyaan": "meow", ...}';
    my $jsonobject = $jsonparser->decode($jsonstring);
    my $messageobj = Sisimai::Message->new('data' => $jsonobject, 'input' => 'json');

If you have implemented a custom MTA module and use it, set the value of "load"
in the argument of this method as an array reference like following code:

    my $messageobj = Sisimai::Message->new(
                        'data'  => $jsonobject,
                        'load'  => ['Your::Custom::MTA::Module']
                        'input' => 'json',
                  );

Beginning from v4.19.0, `hook` argument is available to callback user defined
method like the following codes:

    my $callbackto = sub {
        my $argv = shift;
        my $data = { 'feedback-id' => '', 'account-id' => '' };
        my $mesg = $argv->{'message'} || {};

        if( exists $mesg->{'feedbackId'} ) {
            $data->{'feedback-id'} = $mesg->{'feedback-Id'};
        }

        if( exists $mesg->{'sendingAccountId'} ) {
            $data->{'account-id'} = $mesg->{'sendingAccountId'};
        }
        return $data;
    };
    my $messageobj = Sisimai::Message->new(
                        'data'  => $jsonobject,
                        'hook'  => $callbackto,
                        'input' => 'json' );
    print $message->catch->{'feedback-id'};    # 01010157e48fa03f-c7e948fe-...

=head1 INSTANCE METHODS

=head2 C<B<(from)>>

C<from()> returns empty string

    print $message->from;   # ''

=head2 C<B<header()>>

C<header()> returns empty Hash

    print $message->header; # {}

=head2 C<B<ds()>>

C<ds()> returns an array reference which include contents of delivery status.

    for my $e ( @{ $message->ds } ) {
        print $e->{'status'};   # 5.1.1
        print $e->{'recipient'};# neko@example.jp
    }

=head2 C<B<rfc822()>>

C<rfc822()> returns a hash reference which include the header part of the original
message.

    print $message->rfc822->{'from'};   # cat@example.com
    print $message->rfc822->{'to'};     # neko@example.jp

=head2 C<B<catch()>>

C<catch()> returns any data generated by user-defined method passed at the `hook`
argument of new() constructor.

=head1 AUTHOR

azumakuniyuki

=head1 COPYRIGHT

Copyright (C) 2014-2019 azumakuniyuki, All rights reserved.

=head1 LICENSE

This software is distributed under The BSD 2-Clause License.

=cut

