#
#  Copyright 2009-2013 MongoDB, Inc.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

use strict;
use warnings;
package MongoDB::DBRef;

# ABSTRACT: A MongoDB database reference

use version;
our $VERSION = 'v1.6.0';

use Tie::IxHash;
use Moo;
use MongoDB::_Types qw(
    DBRefColl
    DBRefDB
);
use Types::Standard qw(
    HashRef
    Maybe
);
use namespace::clean -except => 'meta';

#pod =attr id
#pod
#pod Required. The C<_id> value of the referenced document. If the
#pod C<_id> is an ObjectID, then you must use a L<MongoDB::OID> object.
#pod
#pod This may also be specified in the constructor as C<'$id'>.
#pod
#pod =cut

# no type constraint since an _id can be anything
has id => (
    is        => 'ro',
    required  => 1
);

#pod =attr ref
#pod
#pod Required. The collection in which the referenced document lives. Either a
#pod L<MongoDB::Collection> object or a string containing the collection name. The
#pod object will be coerced to string form.
#pod
#pod This may also be specified in the constructor as C<'$ref'>.
#pod
#pod =cut

has ref => (
    is        => 'ro',
    isa       => DBRefColl,
    required  => 1,
    coerce    => DBRefColl->coercion,
);

#pod =attr db
#pod
#pod Optional. The database in which the referenced document lives. Either a
#pod L<MongoDB::Database> object or a string containing the database name. The
#pod object will be coerced to string form.
#pod
#pod Not all other language drivers support the C<$db> field, so using this
#pod field is not recommended.
#pod
#pod This may also be specified in the constructor as C<'$db'>.
#pod
#pod =cut

has db => (
    is        => 'ro',
    isa       => Maybe[DBRefDB],
    coerce    => Maybe([DBRefDB])->coercion,
);

#pod =attr extra
#pod
#pod Optional.  A hash reference of additional fields in the DBRef document.
#pod Not all MongoDB drivers support this feature and you B<should not> rely on
#pod it.  This attribute exists solely to ensure DBRefs generated by drivers that
#pod do allow extra fields will round-trip correctly.
#pod
#pod B<USE OF THIS FIELD FOR NEW DBREFS IS NOT RECOMMENDED.>
#pod
#pod =cut

has extra => (
    is => 'ro',
    isa => HashRef,
    default => sub { {} },
);

around BUILDARGS => sub {
    my $orig  = shift;
    my $class = shift;
    my $hr    = $class->$orig(@_);
    return {
        id => (
              exists( $hr->{'$id'} ) ? delete $hr->{'$id'}
            : exists( $hr->{id} )    ? delete $hr->{id}
            :                          undef
        ),
        ref => (
              exists( $hr->{'$ref'} ) ? delete $hr->{'$ref'}
            : exists( $hr->{ref} )    ? delete $hr->{ref}
            :                           undef
        ),
        db => (
              exists( $hr->{'$db'} ) ? delete $hr->{'$db'}
            : exists( $hr->{db} )    ? delete $hr->{db}
            :                          undef
        ),
        extra => $hr,
    };
};

sub _ordered {
    my $self = shift;

    return Tie::IxHash->new(
        '$ref' => $self->ref,
        '$id'  => $self->id,
        ( defined($self->db) ? ( '$db' => $self->db ) : () ),
        %{ $self->extra },
    );
}


1;

=pod

=encoding UTF-8

=head1 NAME

MongoDB::DBRef - A MongoDB database reference

=head1 VERSION

version v1.6.0

=head1 SYNOPSIS

    my $dbref = MongoDB::DBRef->new(
        ref => 'my_collection',
        id => 123
    );

    $coll->insert( { foo => 'bar', other_doc => $dbref } );

=head1 DESCRIPTION

This module provides support for database references (DBRefs) in the Perl
MongoDB driver. A DBRef is a special embedded document which points to
another document in the database. DBRefs are not the same as foreign keys
and do not provide any referential integrity or constraint checking. For example,
a DBRef may point to a document that no longer exists (or never existed.)

Generally, these are not recommended and "manual references" are preferred.

See L<Database references|http://docs.mongodb.org/manual/reference/database-references/>
en the MongoDB manual for more information.

=head1 ATTRIBUTES

=head2 id

Required. The C<_id> value of the referenced document. If the
C<_id> is an ObjectID, then you must use a L<MongoDB::OID> object.

This may also be specified in the constructor as C<'$id'>.

=head2 ref

Required. The collection in which the referenced document lives. Either a
L<MongoDB::Collection> object or a string containing the collection name. The
object will be coerced to string form.

This may also be specified in the constructor as C<'$ref'>.

=head2 db

Optional. The database in which the referenced document lives. Either a
L<MongoDB::Database> object or a string containing the database name. The
object will be coerced to string form.

Not all other language drivers support the C<$db> field, so using this
field is not recommended.

This may also be specified in the constructor as C<'$db'>.

=head2 extra

Optional.  A hash reference of additional fields in the DBRef document.
Not all MongoDB drivers support this feature and you B<should not> rely on
it.  This attribute exists solely to ensure DBRefs generated by drivers that
do allow extra fields will round-trip correctly.

B<USE OF THIS FIELD FOR NEW DBREFS IS NOT RECOMMENDED.>

=head1 AUTHORS

=over 4

=item *

David Golden <david@mongodb.com>

=item *

Rassi <rassi@mongodb.com>

=item *

Mike Friedman <friedo@friedo.com>

=item *

Kristina Chodorow <k.chodorow@gmail.com>

=item *

Florian Ragwitz <rafl@debian.org>

=back

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2016 by MongoDB, Inc.

This is free software, licensed under:

  The Apache License, Version 2.0, January 2004

=cut

__END__


# vim: set ts=4 sts=4 sw=4 et tw=75:
