# Licensed to the Apache Software Foundation (ASF) under one or more
# contributor license agreements.  See the NOTICE file distributed with
# this work for additional information regarding copyright ownership.
# The ASF licenses this file to You under the Apache License, Version 2.0
# (the "License"); you may not use this file except in compliance with
# the License.  You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

use strict;
use warnings;

package Clownfish::Binding::Core;
use Clownfish::Util qw( verify_args );
use Clownfish::Binding::Core::Function;
use Clownfish::Binding::Core::Method;
use Clownfish::Binding::Core::Class;
use Clownfish::Binding::Core::File;
use Clownfish::Binding::Core::Aliases;

our %new_PARAMS = (
    hierarchy => undef,
    dest      => undef,
    header    => undef,
    footer    => undef,
);

sub new {
    my ( $either, %args ) = @_;
    verify_args( \%new_PARAMS, %args ) or confess $@;
    return _new( @args{qw( hierarchy dest header footer )} );
}

1;

__END__

__POD__

=head1 NAME

Clownfish::Binding::Core - Generate core C code for a Clownfish::Hierarchy.

=head1 SYNOPSIS

    my $hierarchy = Clownfish::Hierarchy->new(
        source => '/path/to/clownfish/files',
        dest   => 'autogen',
    );
    $hierarchy->build;
    my $core_binding = Clownfish::Binding::Core->new(
        hierarchy => $hierarchy,
        dest      => 'autogen',
        header    => "/* Auto-generated file. */\n",
        footer    => $copyfoot,
    );
    my $modified = $core_binding->write_all_modified($modified);

=head1 DESCRIPTION

A Clownfish::Hierarchy describes an abstract specifiction for a class
hierarchy; Clownfish::Binding::Core is responsible for auto-generating C
code which implements that specification.

=head1 METHODS

=head2 new

    my $binding = Clownfish::Binding::Core->new(
        hierarchy => $hierarchy,            # required
        dest      => '/path/to/autogen',    # required
        header    => $header,               # required
        footer    => $footer,               # required
    );

=over

=item * B<hierarchy> - A L<Clownfish::Hierarchy>.

=item * B<dest> - The directory where C output files will be written.

=item * B<header> - Text which will be prepended to each generated C file --
typically, an "autogenerated file" warning.

=item * B<footer> - Text to be appended to the end of each generated C file --
typically copyright information.

=back

=head2 write_all_modified

Call C<< $hierarchy->propagate_modified >> to establish which classes do not
have up-to-date generated .c and .h files, then traverse the hierarchy writing
all necessary files.

=cut

