#!/usr/bin/perl -I/home/phil/perl/cpan/DataTableText/lib
#-------------------------------------------------------------------------------
# Tree operations
# Philip R Brenan at gmail dot com, Appa Apps Ltd Inc., 2020
#-------------------------------------------------------------------------------
# podDocumentation
package Tree::Ops;
our $VERSION = 20200702;
require v5.26;
use warnings FATAL => qw(all);
use strict;
use Carp qw(confess cluck);
use Data::Dump qw(dump);
use Data::Table::Text qw(genHash);
use feature qw(current_sub say);
use experimental qw(smartmatch);

#D1 Build                                                                       # Create a tree.

sub new($)                                                                      #S Create a new child recording the specified user data.
 {my ($user) = @_;                                                              # User data to be recorded in the child
  genHash(__PACKAGE__,                                                          # Child in the tree.
    children   => [],                                                           # Children of this child.
    user       => $user,                                                        # User data for this child.
    parent     => undef,                                                        # Parent for this child.
    lastChild  => undef,                                                        # Last active child chain - enables us to find the currently open scope from the start if the tree.
   );
 }

sub activeScope($)                                                              #P Locate the active scope in a tree.
 {my ($tree) = @_;                                                              # Tree
  my $active;                                                                   # Latest active child
  for(my $l = $tree; $l; $l = $l->lastChild) {$active = $l}                     # Skip down edge of parse tree to deepest active child.
  $active
 }

sub setParentOfChild($$)                                                        #P Set the parent of a child and return the child.
 {my ($child, $parent) = @_;                                                    # Child, parent
  $child->parent = $parent;                                                     # Parent child
  $child
 }

sub open($$)                                                                    # Add a child and make it the currently active scope into which new children will be added.
 {my ($tree, $user) = @_;                                                       # Tree, user data to be recorded in the interior child being opened
  my $parent = activeScope $tree;                                               # Active parent
  my $child  = new $user;                                                       # New child
  push $parent->children->@*, $child;                                           # Place new child last under parent
  $parent->lastChild = $child;                                                  # Make child active
  setParentOfChild $child, $parent                                              # Parent child
 }

sub close($)                                                                    # Close the current scope returning to the previous scope.
 {my ($tree) = @_;                                                              # Tree
  my $parent = activeScope $tree;                                               # Locate active scope
  delete $parent->parent->{lastChild};                                          # Close scope
  $parent
 }

sub single($$)                                                                  # Add one child in the current scope.
 {my ($tree, $user) = @_;                                                       # Tree, user data to be recorded in the child being created
  $tree->open($user);                                                           # Open scope
  $tree->close;                                                                 # Close scope immediately
 }

sub fromLetters($)                                                              # Create a tree from a string of letters - useful for testing.
 {my ($letters) = @_;                                                           # String of letters and ( ).
  my $t = new(my $s = 'a');
  my @l = split //, $letters;
  my @c;
  for my $l(split //, $letters)
   {my $c = shift @c;
    if    ($l eq '(') {$t->open  ($c) if $c}
    elsif ($l eq ')') {$t->single($c) if $c; $t->close}
    else              {$t->single($c) if $c; @c = $l}
   }
  $t
 }

#D1 Navigation                                                                  # Navigate through a tree.

sub first($)                                                                    # Get the first child under the specified parent.
 {my ($parent) = @_;                                                            # Parent
  $parent->children->[0]
 }

sub last($)                                                                     # Get the last child under the specified parent.
 {my ($parent) = @_;                                                            # Parent
  $parent->children->[-1]
 }

sub indexOfChildInParent($)                                                     #P Get the index of a child within the specified parent.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  my $c = $parent->children;                                                    # Siblings
  for(0..$#$c) {return $_ if $$c[$_] == $child}                                 # Locate child and return index
  confess 'Child not found in parent'
 }

sub next($)                                                                     # Get the next sibling following the specified child.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  my $c = $parent->children;                                                    # Siblings
  return undef if @$c == 0 or $$c[-1] == $child;                                # No next child
  $$c[+1 + indexOfChildInParent $child]                                         # Next child
 }

sub prev($)                                                                     # Get the previous sibling of the specified child.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  my $c = $parent->children;                                                    # Siblings
  return undef if @$c == 0 or $$c[0] == $child;                                 # No previous child
  $$c[-1 + indexOfChildInParent $child]                                         # Previous child
 }

sub firstMost($)                                                                # Return the first most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.
 {my ($parent) = @_;                                                            # Child
  my $f;
  for(my $p = $parent; $p; $p = $p->first) {$f = $p}                            # Go first most
  $f
 }

sub lastMost($)                                                                 # Return the last most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.
 {my ($parent) = @_;                                                            # Child
  my $f;
  for(my $p = $parent; $p; $p = $p->last) {$f = $p}                             # Go last most
  $f
 }

#D1 Location                                                                    # Verify the current location.

sub context($)                                                                  # Get the context of the current child.
 {my ($child) = @_;                                                             # Child
  my @c;                                                                        # Context
  for(my $c = $child; $c; $c = $c->parent) {push @c, $c}                        # Walk up
  @c
 }

sub isFirst($)                                                                  # Return the specified child if that child is first under its parent, else return B<undef>.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  $parent->children->[0] == $child ? $child : undef                             # There will be at least one child
 }

sub isLast($)                                                                   # Return the specified child if that child is last under its parent, else return B<undef>.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  my $c = $parent->children;
  $parent->children->[-1] == $child ? $child : undef                            # There will be at least one child
 }

sub singleChildOfParent($)                                                      # Return the only child of this parent if the parent has an only child, else B<undef>
 {my ($parent) = @_;                                                            # Parent
  $parent->children->@* == 1 ? $parent->children->[0] : undef                   # Return only child if it exists
 }

#D1 Put                                                                         # Insert children into a tree.

sub putFirst($$)                                                                # Place a new child first under the specified parent and return the child.
 {my ($parent, $child) = @_;                                                    # Parent, child
  unshift $parent->children->@*, $child;                                        # Place child
  setParentOfChild $child, $parent                                              # Parent child
 }

sub putLast($$)                                                                 # Place a new child last under the specified parent and return the child.
 {my ($parent, $child) = @_;                                                    # Parent, child
  push $parent->children->@*, $child;                                           # Place child
  setParentOfChild $child, $parent                                              # Parent child
 }

sub putNext($$)                                                                 # Place a new child after the specified child.
 {my ($child, $new) = @_;                                                       # Existing child, new child
  return undef unless defined(my $i = indexOfChildInParent $child);             # Locate child within parent
  splice $child->parent->children->@*, $i, 1, $child, $new;                     # Place new child
  setParentOfChild $new, $child->parent                                         # Parent child
 }

sub putPrev($$)                                                                 # Place a new child before the specified child.
 {my ($child, $new) = @_;                                                       # Child, new child
  return undef unless defined(my $i = indexOfChildInParent($child));            # Locate child within parent
  splice $child->parent->children->@*, $i, 1, $new, $child;                     # Place new child
  setParentOfChild $new, $child->parent                                         # Parent child
 }

#D1 Steps                                                                       # Move the start or end of a scope forwards or backwards as suggested by Alex Monroe.

sub step($)                                                                     # Make the first child of the specified parent the parents previous sibling and return the parent. In effect this moves the start of the parent one step forwards.
 {my ($parent) = @_;                                                            # Parent
  return undef unless my $f = $parent->first;                                   # First child
  putPrev $parent, cut $f;                                                      # Place first child
  $parent
 }

sub stepEnd($)                                                                  # Make the next sibling of the specified parent the parents last child and return the parent. In effect this moves the end of the parent one step forwards.
 {my ($parent) = @_;                                                            # Parent
  return undef unless my $n = $parent->next;                                    # Next sibling
  putLast $parent, cut $n;                                                      # Place next sibling as first child
  $parent
 }

sub stepBack                                                                    # Make the previous sibling of the specified parent the parents first child and return the parent. In effect this moves the start of the parent one step backwards.
 {my ($parent) = @_;                                                            # Parent
  return undef unless my $p = $parent->prev;                                    # Previous sibling
  putFirst $parent, cut $p;                                                     # Place previous sibling as first child
  $parent
 }

sub stepEndBack                                                                 # Make the last child of the specified parent the parents next sibling and return the parent. In effect this moves the end of the parent one step backwards.
 {my ($parent) = @_;                                                            # Parent
  return undef unless my $l = $parent->last;                                    # Last child sibling
  putNext $parent, cut $l;                                                      # Place last child as first sibling
  $parent
 }

#D1 Edit                                                                        # Edit a tree in situ.

sub cut($)                                                                      # Cut out a child and all its content and children, return it ready for reinsertion else where.
 {my ($child) = @_;                                                             # Child
  splice $child->parent->children->@*, indexOfChildInParent($child), 1;          # Remove child
  $child
 }

sub dup($)                                                                      # Duplicate a parent and all its descendants.
 {my ($parent) = @_;                                                            # Parent

  sub                                                                           # Duplicate a child
   {my ($old)  = @_;                                                            # Existing child
    my $new    = new $old->user;                                                # New child
    push $new->children->@*, __SUB__->($_) for $old->children->@*;              # Duplicate children of child
    $new
   }->($parent)                                                                 # Start duplication at parent
 }

sub unwrap($)                                                                   # Unwrap the specified child and return that child.
 {my ($child) = @_;                                                             # Child
  return undef unless defined(my $i = indexOfChildInParent $child);             # Locate child within parent
  my $parent = $child->parent;                                                  # Parent
  $_->parent = $parent for $child->children->@*;                                # Reparent unwrapped children of child
  delete $child ->{parent};                                                     # Remove parent of unwrapped child
  splice $parent->children->@*, $i, 1, $child->children->@*;                    # Remove child
  $parent
 }

sub wrap($$)                                                                    # Wrap the specified child with a new parent and return the new parent.
 {my ($child, $user) = @_;                                                      # Child to wrap, user data for new wrapping parent
  return undef unless defined(my $i = indexOfChildInParent $child);             # Locate child within existing parent
  my $parent     = $child->parent;                                              # Existing parent
  my $new        = new $user;                                                   # Create new parent
  $new->parent   = $parent;                                                     # Parent new parent
  $new->children = [$child];                                                    # Set children for new parent
  splice $parent->children->@*, $i, 1, $new;                                    # Place new parent in existing parent
  $child->parent = $new                                                         # Reparent child to new parent
 }

sub merge($)                                                                    # Merge the children of the specified parent with those of the surrounding parents if the L<user> data of those parents L<smartmatch> that of the specified parent. Merged parents are unwrapped. Returns the specified parent regardless. From a proposal made by Micaela Monroe.
 {my ($parent) = @_;                                                            # Merging parent
  while(my $p = $parent->prev)                                                  # Preceding siblings of a parent
   {last unless $p->user ~~ $parent->user;                                      # Preceding parents that carry the same data
    putFirst $parent, cut $p;                                                   # Place merged parent first under merging parent
    unwrap $p;                                                                  # Unwrapped merged parent
   }
  while(my $p = $parent->next)                                                  # Following siblings of a parent
   {last unless $p->user ~~ $parent->user;                                      # Following parents that carry the same data
    putLast $parent, cut $p;                                                    # Place merged parent last under merging parent
    unwrap $p;                                                                  # Unwrap merged parent
   }
  $parent
 }

sub split($)                                                                    # Make the specified parent a grandparent of each of its children by interposing a copy of the specified parent between the specified parent and each of its children. Return the specified parent.
 {my ($parent) = @_;                                                            # Parent to make into a grand parent
  wrap $_, $parent->user for $parent->children->@*;                             # Grandparent each child
  $parent
 }

#D1 Traverse                                                                    # Traverse a tree.

sub by($;$)                                                                     # Traverse a tree in order to process each child with the specified sub and return an array of the results of processing each child. If no sub sub is specified, the children are returned in tree order.
 {my ($tree, $sub) = @_;                                                        # Tree, optional sub to process each child
             $sub //= sub{@_};                                                  # Default sub

  my @r;                                                                        # Results
  sub                                                                           # Traverse
   {my ($child) = @_;                                                           # Child
    __SUB__->($_) for $child->children->@*;                                     # Children of child
    push @r, &$sub($child);                                                     # Process child saving result
   }->($tree);                                                                  # Start at root of tree

  @r
 }

sub select($$)                                                                  # Select matching children in a tree. A child can be selected via named value, array of values, a hash of values, a regular expression or a sub reference.
 {my ($tree, $select) = @_;                                                     # Tree, method to select a child
  my $ref = ref $select;                                                        # Selector type
  my $sel =                                                                     # Selection method
             $ref =~ m(array)i ? sub{grep{$_[0] eq $_} @$select} :              # Array
             $ref =~ m(hash)i  ? sub{$$select{$_[0]}}            :              # Hash
             $ref =~ m(exp)i   ? sub{$_[0] =~ m($select)}        :              # Regular expression
             $ref =~ m(code)i  ? sub{&$select($_[0])}            :              # Sub
                                 sub{$_[0] eq $select};                         # Scalar
  my @s;                                                                        # Selection

  sub                                                                           # Traverse
   {my ($child) = @_;                                                           # Child
    push @s, $child if &$sel($child->user);                                     # Select child if it matches
    __SUB__->($_) for $child->children->@*;                                     # Each child
   }->($tree);                                                                  # Start at root

  @s
 }

#D1 Print                                                                       # Print a tree.

sub print($;$)                                                                  # String representation as a horizontal tree.
 {my ($tree, $print) = @_;                                                      # Tree, optional print method
  my @s;                                                                        # String representation

  sub                                                                           # Print a child
   {my ($child, $depth) = @_;                                                   # Child, depth
    my $user = $child->user;                                                    # User data
    push @s, join '', '  ' x $depth, $print ? &$print($user) : $user;           # Print child
    __SUB__->($_, $depth+1) for $child->children->@*;                           # Print children of child
   }->($tree, 0);                                                               # Print root

  join "\n", @s, ''                                                             # String result
 }

sub brackets($$;$)                                                              # Bracketed string representation of a tree.
 {my ($tree, $print, $separator) = @_;                                          # Tree, print method, child separator
  my @s;                                                                        # String representation
  my $t = $separator // '';                                                     # Default child separator
  sub                                                                           # Print a child
   {my ($child) = @_;                                                           # Child
    my $user = $child->user;                                                    # User data
    my ($p) = ($print ? &$print($user) : $user);                                # Printed child
    my  $c  = $child->children;                                                 # Children of child
    return $p unless @$c;                                                       # Return child immediately if no children to format
    join '', $p, '(', join($t, map {__SUB__->($_)} @$c), ')'                    # String representation
   }->($tree)                                                                   # Print root
 }

#D1 Data Structures                                                             # Data structures use by this package.

#D0
#-------------------------------------------------------------------------------
# Export
#-------------------------------------------------------------------------------

use Exporter qw(import);

use vars qw(@ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

@ISA          = qw(Exporter);
@EXPORT_OK    = qw(
);
%EXPORT_TAGS  = (all=>[@EXPORT, @EXPORT_OK]);

# podDocumentation

=pod

=encoding utf-8

=head1 Name

Tree::Ops - Tree operations.

=head1 Synopsis

Create a tree:

  my $t = Tree::Ops::new 'a';
  for(1..2)
   {$t->open  ('b');
    $t->single('c');
    $t->close;
   }
  $t->single  ('d');

Print the tree:

  is_deeply $t->print(sub{@_}), <<END;
  a
    b
      c
    b
      c
    d
  END

Locate a specific child in the tree and print it:

  is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];

=head1 Description

Tree operations.


Version 20200702.


The following sections describe the methods in each functional area of this
module.  For an alphabetic listing of all methods by name see L<Index|/Index>.



=head1 Build

Create a tree.

=head2 new($user)

Create a new child recording the specified user data.

     Parameter  Description
  1  $user      User data to be recorded in the child

B<Example:>


  if (1)

   {my $t = Tree::Ops::new 'a';  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    for(1..2)
     {$t->open  ('b');
      $t->single('c');
      $t->close;
     }
    $t->single  ('d');
    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END

    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }


This is a static method and so should either be imported or invoked as:

  Tree::Ops::new


=head2 open($tree, $user)

Add a child and make it the currently active scope into which new children will be added.

     Parameter  Description
  1  $tree      Tree
  2  $user      User data to be recorded in the interior child being opened

B<Example:>


  if (1)
   {my $t = Tree::Ops::new 'a';
    for(1..2)

     {$t->open  ('b');  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      $t->single('c');
      $t->close;
     }
    $t->single  ('d');
    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END

    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }


=head2 close($tree)

Close the current scope returning to the previous scope.

     Parameter  Description
  1  $tree      Tree

B<Example:>


  if (1)
   {my $t = Tree::Ops::new 'a';
    for(1..2)
     {$t->open  ('b');
      $t->single('c');

      $t->close;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

     }
    $t->single  ('d');
    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END

    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }


=head2 single($tree, $user)

Add one child in the current scope.

     Parameter  Description
  1  $tree      Tree
  2  $user      User data to be recorded in the child being created

B<Example:>


  if (1)
   {my $t = Tree::Ops::new 'a';
    for(1..2)
     {$t->open  ('b');

      $t->single('c');  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      $t->close;
     }

    $t->single  ('d');  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END

    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }


=head2 fromLetters($letters)

Create a tree from a string of letters - useful for testing.

     Parameter  Description
  1  $letters   String of letters and ( ).

B<Example:>



  ok fromLetters(q(bc(d)))->brackets eq q(a(bc(d)));                                # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head1 Navigation

Navigate through a tree.

=head2 first($parent)

Get the first child under the specified parent.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $c->parent,   $b;

      is_deeply $a->first,    $b;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->last,     $d;
      is_deeply $e->next,     $f;
      is_deeply $f->prev,     $e;


=head2 last($parent)

Get the last child under the specified parent.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $c->parent,   $b;
      is_deeply $a->first,    $b;

      is_deeply $a->last,     $d;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $e->next,     $f;
      is_deeply $f->prev,     $e;


=head2 next($child)

Get the next sibling following the specified child.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $c->parent,   $b;
      is_deeply $a->first,    $b;
      is_deeply $a->last,     $d;

      is_deeply $e->next,     $f;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $f->prev,     $e;


=head2 prev($child)

Get the previous sibling of the specified child.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $c->parent,   $b;
      is_deeply $a->first,    $b;
      is_deeply $a->last,     $d;
      is_deeply $e->next,     $f;

      is_deeply $f->prev,     $e;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head2 firstMost($parent)

Return the first most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.

     Parameter  Description
  1  $parent    Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
      is_deeply $a->print(sub{@_}), <<END;
  a
    b
      c
    y
      x
    y
      x
    d
      e
      f
      g
      h
        i
          j
  END


      is_deeply $a->firstMost->brackets, 'c';  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a-> lastMost->brackets, 'j';


=head2 lastMost($parent)

Return the last most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.

     Parameter  Description
  1  $parent    Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
      is_deeply $a->print(sub{@_}), <<END;
  a
    b
      c
    y
      x
    y
      x
    d
      e
      f
      g
      h
        i
          j
  END

      is_deeply $a->firstMost->brackets, 'c';

      is_deeply $a-> lastMost->brackets, 'j';  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head1 Location

Verify the current location.

=head2 context($child)

Get the context of the current child.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';


      is_deeply [map {$_->user} $x->context], [qw(x y a)];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
      is_deeply join(' ', map{$_->user} $a->by),     "c b x y s t z e f g j i h d a";

      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

      $y->unwrap;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';

      $y = $x->wrap('y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 isFirst($child)

Return the specified child if that child is first under its parent, else return B<undef>.

     Parameter  Description
  1  $child     Child

B<Example:>


    if (1)
     {is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $b->singleChildOfParent, $c;

      is_deeply $e->isFirst, $e;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      ok !$f->isFirst;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      ok !$g->isLast;
      is_deeply $h->isLast, $h;
     }


=head2 isLast($child)

Return the specified child if that child is last under its parent, else return B<undef>.

     Parameter  Description
  1  $child     Child

B<Example:>


    if (1)
     {is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $b->singleChildOfParent, $c;
      is_deeply $e->isFirst, $e;
      ok !$f->isFirst;

      ok !$g->isLast;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      is_deeply $h->isLast, $h;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

     }


=head2 singleChildOfParent($parent)

Return the only child of this parent if the parent has an only child, else B<undef>

     Parameter  Description
  1  $parent    Parent

B<Example:>


    if (1)
     {is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';

      is_deeply $b->singleChildOfParent, $c;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $e->isFirst, $e;
      ok !$f->isFirst;
      ok !$g->isLast;
      is_deeply $h->isLast, $h;
     }


=head1 Put

Insert children into a tree.

=head2 putFirst($parent, $child)

Place a new child first under the specified parent and return the child.

     Parameter  Description
  1  $parent    Parent
  2  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';

      my $z = $b->putNext(new 'z');
      is_deeply $z->brackets, 'z';
      is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';

      my $y = $d->putPrev(new 'y');
      is_deeply $y->brackets, 'y';
      is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';

      $z->putLast(new 't');
      is_deeply $z->brackets, 'z(t)';
      is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';


      $z->putFirst(new 's');  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';


=head2 putLast($parent, $child)

Place a new child last under the specified parent and return the child.

     Parameter  Description
  1  $parent    Parent
  2  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';

      my $z = $b->putNext(new 'z');
      is_deeply $z->brackets, 'z';
      is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';

      my $y = $d->putPrev(new 'y');
      is_deeply $y->brackets, 'y';
      is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';


      $z->putLast(new 't');  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $z->brackets, 'z(t)';
      is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';

      $z->putFirst(new 's');
      is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';


=head2 putNext($child, $new)

Place a new child after the specified child.

     Parameter  Description
  1  $child     Existing child
  2  $new       New child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';


      my $z = $b->putNext(new 'z');  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $z->brackets, 'z';
      is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';

      my $y = $d->putPrev(new 'y');
      is_deeply $y->brackets, 'y';
      is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';

      $z->putLast(new 't');
      is_deeply $z->brackets, 'z(t)';
      is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';

      $z->putFirst(new 's');
      is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';


=head2 putPrev($child, $new)

Place a new child before the specified child.

     Parameter  Description
  1  $child     Child
  2  $new       New child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';

      my $z = $b->putNext(new 'z');
      is_deeply $z->brackets, 'z';
      is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';


      my $y = $d->putPrev(new 'y');  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $y->brackets, 'y';
      is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';

      $z->putLast(new 't');
      is_deeply $z->brackets, 'z(t)';
      is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';

      $z->putFirst(new 's');
      is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';


=head1 Steps

Move the start or end of a scope forwards or backwards as suggested by Alex Monroe.

=head2 step($parent)

Make the first child of the specified parent the parents previous sibling and return the parent. In effect this moves the start of the parent one step forwards.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


      $d->step;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';


      $d->stepBack;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


      $b->stepEnd;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';


      $b->stepEndBack;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 stepEnd($parent)

Make the next sibling of the specified parent the parents last child and return the parent. In effect this moves the end of the parent one step forwards.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

      $d->step;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';

      $d->stepBack;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


      $b->stepEnd;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';


      $b->stepEndBack;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 stepBack()

Make the previous sibling of the specified parent the parents first child and return the parent. In effect this moves the start of the parent one step backwards.


B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

      $d->step;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';


      $d->stepBack;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

      $b->stepEnd;
      is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';

      $b->stepEndBack;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 stepEndBack()

Make the last child of the specified parent the parents next sibling and return the parent. In effect this moves the end of the parent one step backwards.


B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

      $d->step;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';

      $d->stepBack;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

      $b->stepEnd;
      is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';


      $b->stepEndBack;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head1 Edit

Edit a tree in situ.

=head2 cut($child)

Cut out a child and all its content and children, return it ready for reinsertion else where.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';

      is_deeply [map {$_->user} $x->context], [qw(x y a)];

      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
      is_deeply join(' ', map{$_->user} $a->by),     "c b x y s t z e f g j i h d a";


      $z->cut;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

      $y->unwrap;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';

      $y = $x->wrap('y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 dup($parent)

Duplicate a parent and all its descendants.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';

      is_deeply [map {$_->user} $x->context], [qw(x y a)];

      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
      is_deeply join(' ', map{$_->user} $a->by),     "c b x y s t z e f g j i h d a";

      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

      $y->unwrap;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';

      $y = $x->wrap('y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';


      $y->putNext($y->dup);  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 unwrap($child)

Unwrap the specified child and return that child.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';

      is_deeply [map {$_->user} $x->context], [qw(x y a)];

      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
      is_deeply join(' ', map{$_->user} $a->by),     "c b x y s t z e f g j i h d a";

      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';


      $y->unwrap;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';

      $y = $x->wrap('y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 wrap($child, $user)

Wrap the specified child with a new parent and return the new parent.

     Parameter  Description
  1  $child     Child to wrap
  2  $user      User data for new wrapping parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';

      is_deeply [map {$_->user} $x->context], [qw(x y a)];

      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
      is_deeply join(' ', map{$_->user} $a->by),     "c b x y s t z e f g j i h d a";

      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';


      $y->unwrap;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';


      $y = $x->wrap('y');  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 merge($parent)

Merge the children of the specified parent with those of the surrounding parents if the L<user|https://en.wikipedia.org/wiki/User_(computing)> data of those parents L<smartmatch> that of the specified parent. Merged parents are unwrapped. Returns the specified parent regardless. From a proposal made by Micaela Monroe.

     Parameter  Description
  1  $parent    Merging parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

      $d->split;
      is_deeply $d->brackets,               'd(d(e)d(f)d(g)d(h(i(j))))';
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(d(e)d(f)d(g)d(h(i(j)))))';


      $d->first->merge;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $d->brackets,               'd(d(efgh(i(j))))';
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(d(efgh(i(j)))))';

      $d->first->unwrap;
      is_deeply $d->brackets,               'd(efgh(i(j)))';
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 split($parent)

Make the specified parent a grandparent of each of its children by interposing a copy of the specified parent between the specified parent and each of its children. Return the specified parent.

     Parameter  Description
  1  $parent    Parent to make into a grand parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


      $d->split;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $d->brackets,               'd(d(e)d(f)d(g)d(h(i(j))))';
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(d(e)d(f)d(g)d(h(i(j)))))';

      $d->first->merge;
      is_deeply $d->brackets,               'd(d(efgh(i(j))))';
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(d(efgh(i(j)))))';

      $d->first->unwrap;
      is_deeply $d->brackets,               'd(efgh(i(j)))';
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head1 Traverse

Traverse a tree.

=head2 by($tree, $sub)

Traverse a tree in order to process each child with the specified sub and return an array of the results of processing each child. If no sub sub is specified, the children are returned in tree order.

     Parameter  Description
  1  $tree      Tree
  2  $sub       Optional sub to process each child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';

      is_deeply [map {$_->user} $x->context], [qw(x y a)];


      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      is_deeply join(' ', map{$_->user} $a->by),     "c b x y s t z e f g j i h d a";  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

      $y->unwrap;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';

      $y = $x->wrap('y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';


=head2 select($tree, $select)

Select matching children in a tree. A child can be selected via named value, array of values, a hash of values, a regular expression or a sub reference.

     Parameter  Description
  1  $tree      Tree
  2  $select    Method to select a child

B<Example:>


  if (1)
   {my $t = Tree::Ops::new 'a';
    for(1..2)
     {$t->open  ('b');
      $t->single('c');
      $t->close;
     }
    $t->single  ('d');
    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END


    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

   }


=head1 Print

Print a tree.

=head2 print($tree, $print)

String representation as a horizontal tree.

     Parameter  Description
  1  $tree      Tree
  2  $print     Optional print method

B<Example:>


  if (1)
   {my $t = Tree::Ops::new 'a';
    for(1..2)
     {$t->open  ('b');
      $t->single('c');
      $t->close;
     }
    $t->single  ('d');

    is_deeply $t->print, <<END;  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

  a
    b
      c
    b
      c
    d
  END

    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }


=head2 brackets($tree, $print, $separator)

Bracketed string representation of a tree.

     Parameter   Description
  1  $tree       Tree
  2  $print      Print method
  3  $separator  Child separator

B<Example:>



      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲

      is_deeply $a->print(sub{@_}), <<END;
  a
    b
      c
    y
      x
    y
      x
    d
      e
      f
      g
      h
        i
          j
  END


      is_deeply $a->firstMost->brackets, 'c';  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲


      is_deeply $a-> lastMost->brackets, 'j';  # 𝗘𝘅𝗮𝗺𝗽𝗹𝗲



=head1 Data Structures

Data structures use by this package.


=head2 Tree::Ops Definition


Child in the tree.




=head3 Output fields


B<children> - Children of this child.

B<lastChild> - Last active child chain - enables us to find the currently open scope from the start if the tree.

B<parent> - Parent for this child.

B<user> - User data for this child.



=head1 Private Methods

=head2 activeScope($tree)

Locate the active scope in a tree.

     Parameter  Description
  1  $tree      Tree

=head2 setParentOfChild($child, $parent)

Set the parent of a child and return the child.

     Parameter  Description
  1  $child     Child
  2  $parent    Parent

=head2 indexOfChildInParent($child)

Get the index of a child within the specified parent.

     Parameter  Description
  1  $child     Child


=head1 Index


1 L<activeScope|/activeScope> - Locate the active scope in a tree.

2 L<brackets|/brackets> - Bracketed string representation of a tree.

3 L<by|/by> - Traverse a tree in order to process each child with the specified sub and return an array of the results of processing each child.

4 L<close|/close> - Close the current scope returning to the previous scope.

5 L<context|/context> - Get the context of the current child.

6 L<cut|/cut> - Cut out a child and all its content and children, return it ready for reinsertion else where.

7 L<dup|/dup> - Duplicate a parent and all its descendants.

8 L<first|/first> - Get the first child under the specified parent.

9 L<firstMost|/firstMost> - Return the first most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.

10 L<fromLetters|/fromLetters> - Create a tree from a string of letters - useful for testing.

11 L<indexOfChildInParent|/indexOfChildInParent> - Get the index of a child within the specified parent.

12 L<isFirst|/isFirst> - Return the specified child if that child is first under its parent, else return B<undef>.

13 L<isLast|/isLast> - Return the specified child if that child is last under its parent, else return B<undef>.

14 L<last|/last> - Get the last child under the specified parent.

15 L<lastMost|/lastMost> - Return the last most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.

16 L<merge|/merge> - Merge the children of the specified parent with those of the surrounding parents if the L<user|https://en.wikipedia.org/wiki/User_(computing)> data of those parents L<smartmatch> that of the specified parent.

17 L<new|/new> - Create a new child recording the specified user data.

18 L<next|/next> - Get the next sibling following the specified child.

19 L<open|/open> - Add a child and make it the currently active scope into which new children will be added.

20 L<prev|/prev> - Get the previous sibling of the specified child.

21 L<print|/print> - String representation as a horizontal tree.

22 L<putFirst|/putFirst> - Place a new child first under the specified parent and return the child.

23 L<putLast|/putLast> - Place a new child last under the specified parent and return the child.

24 L<putNext|/putNext> - Place a new child after the specified child.

25 L<putPrev|/putPrev> - Place a new child before the specified child.

26 L<select|/select> - Select matching children in a tree.

27 L<setParentOfChild|/setParentOfChild> - Set the parent of a child and return the child.

28 L<single|/single> - Add one child in the current scope.

29 L<singleChildOfParent|/singleChildOfParent> - Return the only child of this parent if the parent has an only child, else B<undef>

30 L<split|/split> - Make the specified parent a grandparent of each of its children by interposing a copy of the specified parent between the specified parent and each of its children.

31 L<step|/step> - Make the first child of the specified parent the parents previous sibling and return the parent.

32 L<stepBack|/stepBack> - Make the previous sibling of the specified parent the parents first child and return the parent.

33 L<stepEnd|/stepEnd> - Make the next sibling of the specified parent the parents last child and return the parent.

34 L<stepEndBack|/stepEndBack> - Make the last child of the specified parent the parents next sibling and return the parent.

35 L<unwrap|/unwrap> - Unwrap the specified child and return that child.

36 L<wrap|/wrap> - Wrap the specified child with a new parent and return the new parent.

=head1 Installation

This module is written in 100% Pure Perl and, thus, it is easy to read,
comprehend, use, modify and install via B<cpan>:

  sudo cpan install Tree::Ops

=head1 Author

L<philiprbrenan@gmail.com|mailto:philiprbrenan@gmail.com>

L<http://www.appaapps.com|http://www.appaapps.com>

=head1 Copyright

Copyright (c) 2016-2019 Philip R Brenan.

This module is free software. It may be used, redistributed and/or modified
under the same terms as Perl itself.

=cut



# Tests and documentation

sub test
 {my $p = __PACKAGE__;
  binmode($_, ":utf8") for *STDOUT, *STDERR;
  return if eval "eof(${p}::DATA)";
  my $s = eval "join('', <${p}::DATA>)";
  $@ and die $@;
  eval $s;
  $@ and die $@;
  1
 }

test unless caller;

1;
# podDocumentation
__DATA__
use warnings FATAL=>qw(all);
use strict;
require v5.26;
use Time::HiRes qw(time);
use Test::More tests => 80;

my $startTime = time();
my $localTest = ((caller(1))[0]//'Tree::Ops') eq "Tree::Ops";                   # Local testing mode
Test::More->builder->output("/dev/null") if $localTest;                         # Suppress output in local testing mode

#goto latestTest;

if (1)                                                                          #Tnew #Topen #Tsingle #Tclose #Tprint #Tselect
 {my $t = Tree::Ops::new 'a';
  for(1..2)
   {$t->open  ('b');
    $t->single('c');
    $t->close;
   }
  $t->single  ('d');
  is_deeply $t->print, <<END;
a
  b
    c
  b
    c
  d
END

  is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
 }

ok fromLetters(q(bc(d)))->brackets eq q(a(bc(d)));                              #TfromLetters

if (1)
 {my $a = Tree::Ops::new('a');  is_deeply $a->brackets, 'a';
  my $b = $a->open      ('b');  is_deeply $b->brackets, 'b';
  my $c = $a->single    ('c');  is_deeply $c->brackets, 'c';
  my $B = $a->close;            is_deeply $B->brackets, 'b(c)'; ok $b == $B;
  my $d = $a->open      ('d');  is_deeply $d->brackets, 'd';
  my $e = $a->single    ('e');  is_deeply $e->brackets, 'e';
  my $f = $a->single    ('f');  is_deeply $f->brackets, 'f';
  my $g = $a->single    ('g');  is_deeply $g->brackets, 'g';
  my $h = $a->open      ('h');  is_deeply $h->brackets, 'h';
  my $i = $a->open      ('i');  is_deeply $i->brackets, 'i';
  my $j = $a->single    ('j');  is_deeply $j->brackets, 'j';

  is_deeply [map {$_->user} $a->select(['b', 'c'])],        ['b', 'c'];
  is_deeply [map {$_->user} $a->select({e=>1})],            ['e'];
  is_deeply [map {$_->user} $a->select(qr(b|d))],            ['b', 'd'];
  is_deeply [map {$_->user} $a->select(sub{$_[0] eq 'c'})], ['c'];

  if (1) {                                                                      #Tparent #Tfirst #Tlast #Tnext #Tprev
    is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
    is_deeply $c->parent,   $b;
    is_deeply $a->first,    $b;
    is_deeply $a->last,     $d;
    is_deeply $e->next,     $f;
    is_deeply $f->prev,     $e;
   }

  is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
  is_deeply $b->parent,  $a;
  is_deeply $c->parent,  $b;
  is_deeply $d->parent,  $a;
  is_deeply $d->first,   $e;
  is_deeply $d->last,    $h;
  is_deeply $e->next,    $f;
  is_deeply $f->prev,    $e;

  ok !$c->first;
  ok !$e->last;
  ok !$h->next;
  ok !$e->prev;

  if (1)                                                                        #TsingleChildOfParent #TisFirst #TisLast
   {is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
    is_deeply $b->singleChildOfParent, $c;
    is_deeply $e->isFirst, $e;
    ok !$f->isFirst;
    ok !$g->isLast;
    is_deeply $h->isLast, $h;
   }

  if (1) {                                                                      #TputFirst #TputLast #TputNext #TputPrev
    is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';

    my $z = $b->putNext(new 'z');
    is_deeply $z->brackets, 'z';
    is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';

    my $y = $d->putPrev(new 'y');
    is_deeply $y->brackets, 'y';
    is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';

    $z->putLast(new 't');
    is_deeply $z->brackets, 'z(t)';
    is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';

    $z->putFirst(new 's');
    is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';
   }

  my ($y, $z) = map {$a->select($_)} 'y', 'z';
  is_deeply $y->brackets, 'y';
  is_deeply $z->brackets, 'z(st)';

  $y->putNext($z->cut);
  is_deeply $a->brackets, 'a(b(c)yz(st)d(efgh(i(j))))';

  my $x = $y->putFirst(new "x");
  is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';

  if (1) {                                                                      #Tcut #Tunwrap #Twrap #Tcontext #Tby #Tdup
    is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';

    is_deeply [map {$_->user} $x->context], [qw(x y a)];

    is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
    is_deeply join(' ', map{$_->user} $a->by),     "c b x y s t z e f g j i h d a";

    $z->cut;
    is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

    $y->unwrap;
    is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';

    $y = $x->wrap('y');
    is_deeply $y->brackets, 'y(x)';
    is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

    $y->putNext($y->dup);
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
   }

  if (1) {                                                                      #Tbrackets #TfirstMost #TlastMost
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
    is_deeply $a->print(sub{@_}), <<END;
a
  b
    c
  y
    x
  y
    x
  d
    e
    f
    g
    h
      i
        j
END

    is_deeply $a->firstMost->brackets, 'c';
    is_deeply $a-> lastMost->brackets, 'j';
   }

  if (1) {                                                                      #Tstep #TstepBack #TstepEnd #TstepEndBack
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

    $d->step;
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';

    $d->stepBack;
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

    $b->stepEnd;
    is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';

    $b->stepEndBack;
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
   }

  if (1) {                                                                      #Tsplit #Tmerge
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

    $d->split;
    is_deeply $d->brackets,               'd(d(e)d(f)d(g)d(h(i(j))))';
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(d(e)d(f)d(g)d(h(i(j)))))';

    $d->first->merge;
    is_deeply $d->brackets,               'd(d(efgh(i(j))))';
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(d(efgh(i(j)))))';

    $d->first->unwrap;
    is_deeply $d->brackets,               'd(efgh(i(j)))';
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
   }
 }

done_testing;

if ($localTest)
 {say "TO finished in ", (time() - $startTime), " seconds";
 }

#   owf(q(/home/phil/z/z/z/zzz.txt), $dfa->dumpAsJson);
