
use JSON::Fast;
use LibCurl::Easy:ver<1.4>;
use Log::Async;

logger.untapped-ok = True;

class Curlie::Response {
  has $.lc handles <statusline content success>;
  has %!status;
  method headers { $!lc.receiveheaders }
  method raw-headers { $!lc.raw-headers };
  method json {
    from-json(self.content);
  }
  method status {
    self.parse-status<status>
  }
  method parse-status {
    %!status = %( <http-version status status-reason> Z=> self.statusline.split(' ') ) unless %!status;
    %!status;
  }
  method gist {
      self.raw-headers
    ~ self.lc.buf.decode
  }
  method Str {
    self.statusline
  }
}

class Curlie {
  has $.lc;
  has $.res;
  has $.opts is rw;
  has $!url;

  my $client;
  sub c is export(:c) {
    $client ||= Curlie.new;
  }

  method new(*%args) {
    my $self = callsame;
    $self.opts = %args;
    $self
  }

  method debug(:$ssl = False, :$curl = False) {
    $!opts<verbose> = 1;
    $!opts<debugfunction> = -> $ez, $type, $buf {
      my $out;
      try {
          $out := $buf.decode.chomp;
          CATCH { default { $out := $buf.gist; } }
      }
      given $type.subst('CURLINFO_','') {
        when 'SSL_DATA_IN'  { debug "< $out" if $ssl; }
        when 'SSL_DATA_OUT' { debug "> $out" if $ssl; }
        when 'HEADER_IN'    { debug "< $_" for $out.lines; }
        when 'HEADER_OUT'   { debug "> $_" for $out.lines; }
        when 'DATA_IN'      { debug "< $_" for $out.lines; }
        when 'DATA_OUT'     { debug "> $_" for $out.lines; }
        when 'TEXT'         { do { debug "* $_" for $out.lines } if $curl }
        default { debug "$type: $out"; }
      }
      True;
		 }
     self;
	}

  method !set-url($url) {
    $!url = $url;
  }

  method !new-request {
    $!lc = LibCurl::Easy.new;
    for $!opts.kv -> $k, $v {
      trace "setting option $k ";
      $!lc.setopt: |($k => $v);
    }
  }

  ## GET

  multi method get($url, Bool :$json, :$headers) {
    self!do-request(:$url, :method<GET>, :$headers, :$json);
  }

  ## POST

  multi method post($url, Pair :$form!, :$headers) {
    self.post($url, form => $form.Hash, :$headers);
  }

  multi method post($url, Bool :$json, Hash(List) :$form, :$headers) {
    self!set-url($url);
    die "json forms not supported" if $json && $form;
    my $body = self!form-data($form) if $form;
    self!do-request(:$url, :method<POST>, :$headers, :$json, :$body);
  }

  multi method post($url, Pair :$json!, :$headers) {
    self.post($url, json => $json.Hash, :$headers)
  }

  multi method post($url, Hash :$json!, :$headers) {
    self!set-url($url);
    my $body = to-json($json);
    self!do-request(:$url, :method<POST>, :$headers, :$body, :json);
  }

  multi method post($url, :$headers, Bool :$json = False, Str:D :$data) {
   self!set-url($url);
   self!do-request($url, $headers, $json);
  }

  method!form-data($form) {
    my $lc = $!lc // LibCurl::Easy.new;
    $form.kv.map( -> $k, $v { "$k=" ~ $lc.escape($v).subst(:g, '%20','+') }).join('&');
  }

  method !do-request(:$url!, :$headers, :$body, Str :$method!, Bool :$json = False) {
    $!url = $url;
    self!new-request;
    $!lc.setopt(URL => $!url);
    if $body {
      $!lc.setopt(postfields => $body);
    }
    given $method {
      $!lc.setopt(:httpget) when 'GET';
      when 'POST' {
        $!lc.setopt: postfields => ($body // "");
      }
    }
    self!set-headers($headers);
    self!set-json-headers if $json;
    try {
      $!lc.perform;
      CATCH {
        default {
          error "$_";
        }
      }
    }
    $!res = Curlie::Response.new: errors => $!, lc => $!lc;
    fail $!res if $! or not $!res.success;
    self;
  }

  method !set-json-headers {
   $!lc.set-header(Accept => 'application/json');
   $!lc.set-header(Content-type => 'application/json');
  }

  method !set-headers($headers) {
   return without $headers;
   my @iter = ($_ ~~ Hash ?? .pairs !! .List) with $headers;
   for @iter -> (:key($k), :value($v)) {
     $!lc.set-header(|($k => "$v"));
   }
  }
}

