/*                                                                
**  Copyright (C) 1996,2007,2010,2019  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/* parameter.h
**
** API to parameter file routines
**/


#include <stdio.h>

#ifndef _PARAMETER_H
#define _PARAMETER_H

#ifdef __cplusplus		/* needed for C++ code */
extern "C"
{
#endif

#include <cxcparam/parambind.h>



#ifndef PARAMETER_FILE_SOURCE
  typedef void *paramfile;	/* Make the Param data types opaque */
  typedef void *pmatchlist;
  typedef paramfile ParamFile;
  typedef pmatchlist ParamMatchList;

#else
#include <cxcparam/ptemplat.h>
  typedef ParamFile paramfile;
  typedef ParamMatchList *pmatchlist;
#endif




  extern paramfile PFFile;	/* Global parameter file            */
  extern paramfile PFList;	/* Global parameter file list       */
#ifndef INDEFS
#define INDEFS          (-32767)
#endif
#if __alpha && __unix__
#ifndef INDEFL
#define INDEFL          (-2147483647)
#endif
#ifndef INDEFI
#define INDEFI          (-2147483647)
#endif
#else
#ifndef INDEFL
#define INDEFL          (-2147483647)
#endif
#ifndef INDEFI
#define INDEFI          INDEFL
#endif
#endif
#ifndef INDEFR
#define INDEFR          1.6e38
#endif
#ifndef INDEFD
#define INDEFD          1.6e38
#endif
#ifndef INDEF
#define INDEF           INDEFR
#endif

/* Routines to manipulate parameter files

  paramopen	Open a parameter file and return a paramfile handle

	The argc and argv arguments are the C main's arguments.  If
	you don't want the command line arguments folded into the 
	parameter file pass them as NULL and 0.
	
  paramfind	Search for a parameter file and return its full path

	The extn and path arguments are lists of allowable extension and 
	directories to attempt to search for parameter files.  The default
	search in paramopen is:

	extn = ".par .rdb $PFEXTN"
	path = "$PDIRS $PFILES $UPARM"

	Identifiers prefixed with "$" are recursivly expanded in the run
	time environment.  For comaptiblity with IRAF $UPARM should be set 
	to a single directory name with a trailing "/".  $PFILES may be
	set to a space or "," separated list of directories.

  paramclose	Close the paramfile handle

**/

  char *evaluateIndir (paramfile pf, char *name, char *val);

  paramfile paramopen (const char *filename, char **argv, int argc,
		       const char *mode);
  char *paramfind (const char *, char *, char *, char *);
  void paramclose (paramfile pfile);
  void paramunlock (paramfile pfile);
  void paramlist (paramfile pfile);

  char *paramgetpath (paramfile pfile);	/* returns path of current param file */

/* Error handling
**/
  void paramerr (int, const char *, const char *);	/* Report an error to the user  */
  char *paramerrstr (void);	/* Return the current error string     */
  typedef void (*paramerrvector) (int, const char *, const char *);
  paramerrvector paramerract (paramerrvector newact);


/* Check if a string type value is valid for this parameter
*/
  int paramvalid (paramfile pfile, char *pname, char *pvalue, char *cvalue);
  int paccess (paramfile pfile, const char *pname);

/* Get  parameter values from the parameter file.
**/
  int pgetb (paramfile pfile, const char *pname);
  short pgets (paramfile pfile, const char *pname);
  int pgeti (paramfile pfile, const char *pname);
  long pgetl (paramfile pfile, const char *pname);
  float pgetf (paramfile pfile, const char *pname);
  double pgetd (paramfile pfile, const char *pname);
  char *pgetstr (paramfile pfile, const char *pname, char *string, int length);


/* Put parameter values into the parameter file.
**/
  void pputb (paramfile pfile, const char *pname, int bvalue);
  void pputs (paramfile pfile, const char *pname, short svalue);
  void pputi (paramfile pfile, const char *pname, int ivalue);
  void pputl (paramfile pfile, const char *pname, long lvalue);
  void pputf (paramfile pfile, const char *pname, float fvalue);
  void pputd (paramfile pfile, const char *pname, double dvalue);
  void pputstr (paramfile pfile, const char *pname, char *string);


/* Read a list of parameter names from the parameter file
**
	pmatchopen builds a list of parameter names which match the
	supplied template.  pmatchnext returns each name in turn on
	sucessive calls.

	template meta-characters:

	?   		match any character, but there must be one
	*		match anything, or nothing
	[<c>...]	match an inclusive set

**/
  pmatchlist pmatchopen (paramfile pfile, char *ptemplate);
  char *pmatchnext (pmatchlist mlist);
  int pmatchlength (pmatchlist mlist);
  void pmatchrewind (pmatchlist mlist);
  void pmatchclose (pmatchlist mlist);
  char *pfilelast (void);

#include <cxcparam/parament.h>

/* The default parameter file interface
**/
/********* REVERSED to order used by pre-existing tools    wlm 06/09/98 
#define clinit(argc, argv, mode)  ( (PFFile = paramopen(NULL, argv, argc, mode)) == NULL ? paramerr(1, "clinit", pfilelast()), PFFile : PFFile)
#define clinit(argv, argc, mode)  ( (PFFile = paramopen(NULL, argv, argc, mode)) == NULL ? paramerr(1, "clinit", pfilelast()), PFFile : PFFile)
**********/

#define clinit(argv, argc, mode)  (PFFile = paramopen(NULL, argv, argc, mode))

#define clgetb(pname)			pgetb(PFFile, pname)
#define clgets(pname)			pgets(PFFile, pname)
#define clgeti(pname)			pgeti(PFFile, pname)
#define clgetf(pname)			pgetf(PFFile, pname)
#define clgetd(pname)			pgetd(PFFile, pname)
#define clgetstr(pname, string, max)    pgetstr(PFFile, pname, string, max)
#define clgstr(pname, string, max)	pgetstr(PFFile, pname, string, max)

#define clputb(pname, value)		pputb(PFFile, pname, value)
#define clputs(pname, value)		pputs(PFFile, pname, value)
#define clputi(pname, value)		pputi(PFFile, pname, value)
#define clputf(pname, value)		pputf(PFFile, pname, value)
#define clputd(pname, value)		pputd(PFFile, pname, value)
#define clputstr(pname, string)		pputstr(PFFile, pname, string)
#define clpstr(pname, string)		pputstr(PFFile, pname, string)

  int pgetenum (paramfile pfile, char *pname, char *dict);

#define clgetenum(pname, dict)                  pgetenum(PFFile, pname, dict)

/* to turn off exit on error flag  02/16/99 */
  int set_paramerror (int new_val);

#define clclose() paramclose(PFFile)


/* pp here is of type paramfile
 * so should be declared in the
 * calling code as..
 *
 * paramfile pp;
 */

#define clopset(pset, mode)  (paramopen(pset, NULL, 0, mode))
#define clcpset(pp)                     paramclose(pp)

#define clgpsetb(pp, param)             pgetb(pp, param)
#define clgpsetc(pp, param)             pgetc(pp, param)
#define clgpsets(pp, param)             pgets(pp, param)
#define clgpseti(pp, param)             pgeti(pp, param)
#define clgpsetl(pp, param)             pgetl(pp, param)
#define clgpsetr(pp, param)             pgetr(pp, param)
#define clgpsetd(pp, param)             pgetd(pp, param)
#define clgpsetx(pp, param)             pgetx(pp, param)
#define clgpset(pp, pname, string, max)     pgetstr(pp, pname, string, max)

#define clppsetb(pp, param, pval)       pputb(pp, param, pval)
#define clppsetc(pp, param, pval)       pputc(pp, param, pval)
#define clppsets(pp, param, pval)       pputs(pp, param, pval)
#define clppseti(pp, param, pval)       pputi(pp, param, pval)
#define clppsetl(pp, param, pval)       pputl(pp, param, pval)
#define clppsetr(pp, param, pval)       pputr(pp, param, pval)
#define clppsetd(pp, param, pval)       pputd(pp, param, pval)
#define clppsetx(pp, param, pval)       pputx(pp, param, pval)
#define clppset(pp, pname, string)	pputstr(pp, pname, string)

#define PARAMTOOLNAMELEN  100

  extern char paramToolName[PARAMTOOLNAMELEN];

#ifndef NONCIAO
  int paramHelpFunction (char *, char *, char *);
#endif

#ifndef SZ_PFLINE
#define SZ_PFLINE	2048
#endif

#ifdef __cplusplus
}
#endif

#endif
