package Graphics::Grid::Grill;

# ABSTRACT: Grill grob

use Graphics::Grid::Class;

our $VERSION = '0.001'; # VERSION

use Graphics::Grid::GPar;
use Graphics::Grid::Types qw(:all);
use Graphics::Grid::Unit;


has [qw(h v)] => (
    is      => 'ro',
    isa     => UnitLike,
    coerce  => 1,
    default => sub { Graphics::Grid::Unit->new( [ 0.25, 0.5, 0.75 ] ); },
);


with qw(
  Graphics::Grid::Grob
);

has '+gp' => ( default => sub { Graphics::Grid::GPar->new( col => "grey" ) }, );

method _build_elems () { 1; }

method _draw ($grid) {
    my $make_line_unit = fun($h_or_v, $idx) {
        my $u = $h_or_v->at($idx);
        return Graphics::Grid::Unit->new([($u->value->[0]) x 2], $u->unit->[0]);
    };

    my @lines = (
        (
            map {
                Graphics::Grid::Grob::Lines->new(
                    x  => [ 0,  1 ],
                    y  => $make_line_unit->($self->h, $_),
                    gp => $self->gp,
                  )
            } ( 0 .. $self->h->elems - 1 )
        ),
        (
            map {
                Graphics::Grid::Grob::Lines->new(
                    x  => $make_line_unit->($self->v, $_),
                    y  => [ 0,  1 ],
                    gp => $self->gp,
                  )
            } ( 0 .. $self->v->elems - 1 )
        ),
    );

    for (@lines) {
        $_->_draw($grid);
    }
}

__PACKAGE__->meta->make_immutable;

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Graphics::Grid::Grill - Grill grob

=head1 VERSION

version 0.001

=head1 SYNOPSIS

    use Graphics::Grid::Grill;
    use Graphics::Grid::GPar;
    my $rect = Graphics::Grid::Grill->new(
            h => unit([0.25, 0.5, 0.75]),
            v => unit([0.25, 0.5, 0.75]),
            gp => Graphics::Grid::GPar->new(col => "grey"));

    # or use the function interface
    use Graphics::Grid::Functions qw(:all);
    my $rect = grill(%params);

=head1 DESCRIPTION

This class represents a grill graphical object.    

=head1 ATTRIBUTES

=head2 h

A unit object indicating the horizontal location of the vertical grill lines.
Default is C<unit([0.25, 0.5, 0.75])>.

=head2 v

A unit object indicating the vertical location of the horizontal grill lines.
Default is C<unit([0.25, 0.5, 0.75])>.

=head2 gp

An object of L<Graphics::Grid::GPar>. Default is an empty gpar object.

Default is C<gpar(col =E<gt> "grey")>.

=head2 vp

A viewport object. When drawing a grob, if the grob has this attribute, the
viewport would be temporily pushed onto the global viewport stack before drawing
takes place, and be poped after drawing. If the grob does not have this attribute
set, it would be drawn on the existing current viewport in the global viewport
stack. 

=head1 SEE ALSO

L<Graphics::Grid::Functions>

L<Graphics::Grid::Grob>

=head1 AUTHOR

Stephan Loyd <sloyd@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2018-2023 by Stephan Loyd.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
