/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.blob.file;

import com.github.fge.lambdas.Throwing;
import com.google.common.base.Preconditions;
import com.google.common.io.ByteSource;
import jakarta.inject.Inject;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.time.Duration;
import java.util.Collection;
import org.apache.commons.io.FileUtils;
import org.apache.james.blob.api.BlobId;
import org.apache.james.blob.api.BlobStoreDAO;
import org.apache.james.blob.api.BucketName;
import org.apache.james.blob.api.ObjectNotFoundException;
import org.apache.james.blob.api.ObjectStoreIOException;
import org.apache.james.filesystem.api.FileSystem;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;
import reactor.util.retry.Retry;

public class FileBlobStoreDAO
implements BlobStoreDAO {
    private final File root;
    private final BlobId.Factory blobIdFactory;

    @Inject
    public FileBlobStoreDAO(FileSystem fileSystem, BlobId.Factory blobIdFactory) throws FileNotFoundException {
        this.root = fileSystem.getFile("file://var/blob");
        this.blobIdFactory = blobIdFactory;
    }

    public InputStream read(BucketName bucketName, BlobId blobId) throws ObjectStoreIOException, ObjectNotFoundException {
        File bucketRoot = this.getBucketRoot(bucketName);
        File blob = new File(bucketRoot, blobId.asString());
        try {
            return new FileInputStream(blob);
        }
        catch (FileNotFoundException e) {
            throw new ObjectNotFoundException(String.format("Cannot locate %s within %s", blobId.asString(), bucketName.asString()), (Throwable)e);
        }
    }

    private File getBucketRoot(BucketName bucketName) {
        File bucketRoot = new File(this.root, bucketName.asString());
        if (!bucketRoot.exists()) {
            try {
                FileUtils.forceMkdir((File)bucketRoot);
            }
            catch (IOException e) {
                throw new ObjectStoreIOException("Cannot create bucket", (Throwable)e);
            }
        }
        return bucketRoot;
    }

    public Mono<InputStream> readReactive(BucketName bucketName, BlobId blobId) {
        return Mono.fromCallable(() -> this.read(bucketName, blobId)).subscribeOn(Schedulers.boundedElastic());
    }

    public Mono<byte[]> readBytes(BucketName bucketName, BlobId blobId) {
        return Mono.fromCallable(() -> {
            File bucketRoot = this.getBucketRoot(bucketName);
            File blob = new File(bucketRoot, blobId.asString());
            return FileUtils.readFileToByteArray((File)blob);
        }).onErrorResume(NoSuchFileException.class, e -> Mono.error((Throwable)new ObjectNotFoundException(String.format("Cannot locate %s within %s", blobId.asString(), bucketName.asString()), (Throwable)e))).subscribeOn(Schedulers.boundedElastic());
    }

    public Mono<Void> save(BucketName bucketName, BlobId blobId, byte[] data) {
        Preconditions.checkNotNull((Object)data);
        return Mono.fromRunnable(() -> {
            File bucketRoot = this.getBucketRoot(bucketName);
            File blob = new File(bucketRoot, blobId.asString());
            this.save(data, blob);
        }).subscribeOn(Schedulers.boundedElastic()).then();
    }

    public Mono<Void> save(BucketName bucketName, BlobId blobId, InputStream inputStream) {
        Preconditions.checkNotNull((Object)inputStream);
        return Mono.fromRunnable(() -> {
            File bucketRoot = this.getBucketRoot(bucketName);
            File blob = new File(bucketRoot, blobId.asString());
            this.save(inputStream, blob);
        }).subscribeOn(Schedulers.boundedElastic()).then().retryWhen((Retry)Retry.backoff((long)10L, (Duration)Duration.ofMillis(100L)).filter(e -> e instanceof OverlappingFileLockException));
    }

    private void save(InputStream inputStream, File blob) {
        if (blob.exists()) {
            return;
        }
        try (FileOutputStream out = new FileOutputStream(blob);
             FileChannel channel = out.getChannel();
             FileLock fileLock = channel.lock();){
            inputStream.transferTo(out);
        }
        catch (IOException e) {
            throw new ObjectStoreIOException("IOException occured", (Throwable)e);
        }
    }

    private void save(byte[] data, File blob) {
        if (blob.exists()) {
            return;
        }
        try (FileOutputStream out = new FileOutputStream(blob);
             FileChannel channel = out.getChannel();
             FileLock fileLock = channel.lock();){
            out.write(data);
        }
        catch (IOException e) {
            throw new ObjectStoreIOException("IOException occured", (Throwable)e);
        }
    }

    public Mono<Void> save(BucketName bucketName, BlobId blobId, ByteSource content) {
        return Mono.fromCallable(() -> {
            try {
                return content.read();
            }
            catch (IOException e) {
                throw new ObjectStoreIOException("IOException occured", (Throwable)e);
            }
        }).flatMap(bytes -> this.save(bucketName, blobId, (byte[])bytes));
    }

    public Mono<Void> delete(BucketName bucketName, BlobId blobId) {
        Preconditions.checkNotNull((Object)bucketName);
        return Mono.fromRunnable((Runnable)Throwing.runnable(() -> {
            File bucketRoot = this.getBucketRoot(bucketName);
            File blob = new File(bucketRoot, blobId.asString());
            FileUtils.deleteQuietly((File)blob);
        })).subscribeOn(Schedulers.boundedElastic()).then();
    }

    public Publisher<Void> delete(BucketName bucketName, Collection<BlobId> blobIds) {
        return Flux.fromIterable(blobIds).flatMap(id -> this.delete(bucketName, (BlobId)id)).then();
    }

    public Mono<Void> deleteBucket(BucketName bucketName) {
        return Mono.fromRunnable((Runnable)Throwing.runnable(() -> {
            File bucketRoot = new File(this.root, bucketName.asString());
            FileUtils.deleteQuietly((File)bucketRoot);
        })).subscribeOn(Schedulers.boundedElastic()).then();
    }

    public Publisher<BucketName> listBuckets() {
        return Mono.fromCallable(() -> Files.list(this.root.toPath())).flatMapMany(Flux::fromStream).map(path -> BucketName.of((String)path.getFileName().toString())).subscribeOn(Schedulers.boundedElastic()).onErrorResume(NoSuchFileException.class, e -> Flux.empty());
    }

    public Publisher<BlobId> listBlobs(BucketName bucketName) {
        return Mono.fromCallable(() -> {
            File bucketRoot = this.getBucketRoot(bucketName);
            return Files.list(bucketRoot.toPath());
        }).flatMapMany(Flux::fromStream).map(path -> this.blobIdFactory.parse(path.getFileName().toString())).subscribeOn(Schedulers.boundedElastic());
    }
}

