/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.spark.sql.connector.catalog;

import org.apache.spark.annotation.Evolving;
import org.apache.spark.sql.connector.expressions.Transform;
import org.apache.spark.sql.types.DataType;

/**
 * Interface for a metadata column.
 * <p>
 * A metadata column can expose additional metadata about a row. For example, rows from Kafka can
 * use metadata columns to expose a message's topic, partition number, and offset.
 * <p>
 * A metadata column could also be the result of a transform applied to a value in the row. For
 * example, a partition value produced by bucket(id, 16) could be exposed by a metadata column. In
 * this case, {@link #transform()} should return a non-null {@link Transform} that produced the
 * metadata column's values.
 *
 * @since 3.1.0
 */
@Evolving
public interface MetadataColumn {
  /**
   * Indicates whether a row-level operation should preserve the value of the metadata column
   * for deleted rows. If set to true, the metadata value will be retained and passed back to
   * the writer. If false, the metadata value will be replaced with {@code null}.
   * <p>
   * This flag applies only to row-level operations working with deltas of rows. Group-based
   * operations handle deletes by discarding matching records.
   *
   * @since 4.0.0
   */
  String PRESERVE_ON_DELETE = "__preserve_on_delete";
  boolean PRESERVE_ON_DELETE_DEFAULT = true;

  /**
   * Indicates whether a row-level operation should preserve the value of the metadata column
   * for updated rows. If set to true, the metadata value will be retained and passed back to
   * the writer. If false, the metadata value will be replaced with {@code null}.
   * <p>
   * This flag applies to both group-based and delta-based row-level operations.
   *
   * @since 4.0.0
   */
  String PRESERVE_ON_UPDATE = "__preserve_on_update";
  boolean PRESERVE_ON_UPDATE_DEFAULT = true;

  /**
   * Indicates whether a row-level operation should preserve the value of the metadata column
   * for reinserted rows generated by splitting updates into deletes and inserts. If true,
   * the metadata value will be retained and passed back to the writer. If false, the metadata
   * value will be replaced with {@code null}.
   * <p>
   * This flag applies only to row-level operations working with deltas of rows. Group-based
   * operations do not represent updates as deletes and inserts.
   *
   * @since 4.0.0
   */
  String PRESERVE_ON_REINSERT = "__preserve_on_reinsert";
  boolean PRESERVE_ON_REINSERT_DEFAULT = false;

  /**
   * The name of this metadata column.
   *
   * @return a String name
   */
  String name();

  /**
   * The data type of values in this metadata column.
   *
   * @return a {@link DataType}
   */
  DataType dataType();

  /**
   * @return whether values produced by this metadata column may be null
   */
  default boolean isNullable() {
    return true;
  }

  /**
   * Documentation for this metadata column, or null.
   *
   * @return a documentation String
   */
  default String comment() {
    return null;
  }

  /**
   * The {@link Transform} used to produce this metadata column from data rows, or null.
   *
   * @return a {@link Transform} used to produce the column's values, or null if there isn't one
   */
  default Transform transform() {
    return null;
  }

  /**
   * Returns the column metadata in JSON format.
   *
   * @since 4.0.0
   */
  default String metadataInJSON() {
    return null;
  }
}
