// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/public/cpp/accelerator_actions.h"

#include <string>

#include "base/check_op.h"
#include "base/containers/fixed_flat_map.h"

namespace ash {

constexpr static auto kAcceleratorActionToName = base::MakeFixedFlatMap<
    AcceleratorAction,
    const char*>({
    {AcceleratorAction::kBrightnessDown, "BrightnessDown"},
    {AcceleratorAction::kBrightnessUp, "BrightnessUp"},
    {AcceleratorAction::kCycleBackwardMru, "CycleBackwardMru"},
    {AcceleratorAction::kCycleForwardMru, "CycleForwardMru"},
    {AcceleratorAction::kCycleSameAppWindowsBackward,
     "CycleSameAppWindowsBackward"},
    {AcceleratorAction::kCycleSameAppWindowsForward,
     "CycleSameAppWindowsForward"},
    {AcceleratorAction::kDesksActivateDeskLeft, "DesksActivateDeskLeft"},
    {AcceleratorAction::kDesksActivateDeskRight, "DesksActivateDeskRight"},
    {AcceleratorAction::kDesksMoveActiveItemLeft, "DesksMoveActiveItemLeft"},
    {AcceleratorAction::kDesksMoveActiveItemRight, "DesksMoveActiveItemRight"},
    {AcceleratorAction::kDesksNewDesk, "DesksNewDesk"},
    {AcceleratorAction::kDesksRemoveCurrentDesk, "DesksRemoveCurrentDesk"},
    {AcceleratorAction::kDesksActivate0, "DesksActivate0"},
    {AcceleratorAction::kDesksActivate1, "DesksActivate1"},
    {AcceleratorAction::kDesksActivate2, "DesksActivate2"},
    {AcceleratorAction::kDesksActivate3, "DesksActivate3"},
    {AcceleratorAction::kDesksActivate4, "DesksActivate4"},
    {AcceleratorAction::kDesksActivate5, "DesksActivate5"},
    {AcceleratorAction::kDesksActivate6, "DesksActivate6"},
    {AcceleratorAction::kDesksActivate7, "DesksActivate7"},
    {AcceleratorAction::kDesksToggleAssignToAllDesks,
     "DesksToggleAssignToAllDesks"},
    {AcceleratorAction::kDisableCapsLock, "DisableCapsLock"},
    {AcceleratorAction::kEnableOrToggleDictation, "EnableOrToggleDictation"},
    {AcceleratorAction::kExit, "Exit"},
    {AcceleratorAction::kFocusCameraPreview, "FocusCameraPreview"},
    {AcceleratorAction::kFocusNextPane, "FocusNextPane"},
    {AcceleratorAction::kFocusPreviousPane, "FocusPreviousPane"},
    {AcceleratorAction::kFocusShelf, "FocusShelf"},
    {AcceleratorAction::kFocusPip, "FocusPip"},
    {AcceleratorAction::kKeyboardBacklightToggle, "KeyboardBacklightToggle"},
    {AcceleratorAction::kKeyboardBrightnessDown, "KeyboardBrightnessDown"},
    {AcceleratorAction::kKeyboardBrightnessUp, "KeyboardBrightnessUp"},
    {AcceleratorAction::kLaunchApp0, "LaunchApp0"},
    {AcceleratorAction::kLaunchApp1, "LaunchApp1"},
    {AcceleratorAction::kLaunchApp2, "LaunchApp2"},
    {AcceleratorAction::kLaunchApp3, "LaunchApp3"},
    {AcceleratorAction::kLaunchApp4, "LaunchApp4"},
    {AcceleratorAction::kLaunchApp5, "LaunchApp5"},
    {AcceleratorAction::kLaunchApp6, "LaunchApp6"},
    {AcceleratorAction::kLaunchApp7, "LaunchApp7"},
    {AcceleratorAction::kLaunchLastApp, "LaunchLastApp"},
    {AcceleratorAction::kLockPressed, "LockPressed"},
    {AcceleratorAction::kLockReleased, "LockReleased"},
    {AcceleratorAction::kLockScreen, "LockScreen"},
    {AcceleratorAction::kMagnifierZoomIn, "MagnifierZoomIn"},
    {AcceleratorAction::kMagnifierZoomOut, "MagnifierZoomOut"},
    {AcceleratorAction::kMediaFastForward, "MediaFastForward"},
    {AcceleratorAction::kMediaNextTrack, "MediaNextTrack"},
    {AcceleratorAction::kMediaPause, "MediaPause"},
    {AcceleratorAction::kMediaPlay, "MediaPlay"},
    {AcceleratorAction::kMediaPlayPause, "MediaPlayPause"},
    {AcceleratorAction::kMediaPrevTrack, "MediaPrevTrack"},
    {AcceleratorAction::kMediaRewind, "MediaRewind"},
    {AcceleratorAction::kMediaStop, "MediaStop"},
    {AcceleratorAction::kMicrophoneMuteToggle, "MicrophoneMuteToggle"},
    {AcceleratorAction::kMoveActiveWindowBetweenDisplays,
     "MoveActiveWindowBetweenDisplays"},
    {AcceleratorAction::kNewIncognitoWindow, "NewIncognitoWindow"},
    {AcceleratorAction::kNewTab, "NewTab"},
    {AcceleratorAction::kNewWindow, "NewWindow"},
    {AcceleratorAction::kOpenCalculator, "OpenCalculator"},
    {AcceleratorAction::kOpenCrosh, "OpenCrosh"},
    {AcceleratorAction::kOpenDiagnostics, "OpenDiagnostics"},
    {AcceleratorAction::kOpenFeedbackPage, "OpenFeedbackPage"},
    {AcceleratorAction::kOpenFileManager, "OpenFileManager"},
    {AcceleratorAction::kOpenGetHelp, "OpenGetHelp"},
    {AcceleratorAction::kPasteClipboardHistoryPlainText,
     "PasteClipboardHistoryPlainText"},
    {AcceleratorAction::kPowerPressed, "PowerPressed"},
    {AcceleratorAction::kPowerReleased, "PowerReleased"},
    {AcceleratorAction::kPrintUiHierarchies, "PrintUiHierarchies"},
    {AcceleratorAction::kPrivacyScreenToggle, "PrivacyScreenToggle"},
    {AcceleratorAction::kRestoreTab, "RestoreTab"},
    {AcceleratorAction::kRotateScreen, "RotateScreen"},
    {AcceleratorAction::kRotateWindow, "RotateWindow"},
    {AcceleratorAction::kScaleUiDown, "ScaleUiDown"},
    {AcceleratorAction::kScaleUiReset, "ScaleUiReset"},
    {AcceleratorAction::kScaleUiUp, "ScaleUiUp"},
    {AcceleratorAction::kShowEmojiPicker, "ShowEmojiPicker"},
    {AcceleratorAction::kToggleImeMenuBubble, "ToggleImeMenuBubble"},
    {AcceleratorAction::kShowShortcutViewer, "ShowShortcutViewer"},
    {AcceleratorAction::kShowTaskManager, "ShowTaskManager"},
    {AcceleratorAction::kStartAssistant, "StartAssistant"},
    {AcceleratorAction::kStopScreenRecording, "StopScreenRecording"},
    {AcceleratorAction::kSuspend, "Suspend"},
    {AcceleratorAction::kSwapPrimaryDisplay, "SwapPrimaryDisplay"},
    {AcceleratorAction::kSwitchIme, "SwitchIme"},
    {AcceleratorAction::kSwitchToLastUsedIme, "SwitchToLastUsedIme"},
    {AcceleratorAction::kSwitchToNextIme, "SwitchToNextIme"},
    {AcceleratorAction::kSwitchToNextUser, "SwitchToNextUser"},
    {AcceleratorAction::kSwitchToPreviousUser, "SwitchToPreviousUser"},
    {AcceleratorAction::kTakePartialScreenshot, "TakePartialScreenshot"},
    {AcceleratorAction::kTakeScreenshot, "TakeScreenshot"},
    {AcceleratorAction::kTakeWindowScreenshot, "TakeWindowScreenshot"},
    {AcceleratorAction::kToggleAppList, "ToggleAppList"},
    {AcceleratorAction::kToggleCalendar, "ToggleCalendar"},
    {AcceleratorAction::kToggleCapsLock, "ToggleCapsLock"},
    {AcceleratorAction::kToggleClipboardHistory, "ToggleClipboardHistory"},
    {AcceleratorAction::kToggleDockedMagnifier, "ToggleDockedMagnifier"},
    {AcceleratorAction::kToggleFloating, "ToggleFloating"},
    {AcceleratorAction::kToggleFullscreen, "ToggleFullscreen"},
    {AcceleratorAction::kToggleFullscreenMagnifier,
     "ToggleFullscreenMagnifier"},
    {AcceleratorAction::kToggleGameDashboard, "ToggleGameDashboard"},
    {AcceleratorAction::kToggleHighContrast, "ToggleHighContrast"},
    {AcceleratorAction::kToggleMaximized, "ToggleMaximized"},
    {AcceleratorAction::kToggleMessageCenterBubble,
     "ToggleMessageCenterBubble"},
    {AcceleratorAction::kToggleMirrorMode, "ToggleMirrorMode"},
    {AcceleratorAction::kToggleMultitaskMenu, "ToggleMultitaskMenu"},
    {AcceleratorAction::kToggleOverview, "ToggleOverview"},
    {AcceleratorAction::kToggleProjectorMarker, "ToggleProjectorMarker"},
    {AcceleratorAction::kToggleResizeLockMenu, "ToggleResizeLockMenu"},
    {AcceleratorAction::kToggleSnapGroup, "ToggleSnapGroup"},
    {AcceleratorAction::kToggleSnapGroupWindowsMinimizeAndRestore,
     "ToggleSnapGroupWindowsMinimizeAndRestore"},
    {AcceleratorAction::kToggleSpokenFeedback, "ToggleSpokenFeedback"},
    {AcceleratorAction::kToggleStylusTools, "ToggleStylusTools"},
    {AcceleratorAction::kToggleSystemTrayBubble, "ToggleSystemTrayBubble"},
    {AcceleratorAction::kToggleWifi, "ToggleWifi"},
    {AcceleratorAction::kTouchHudClear, "TouchHudClear"},
    {AcceleratorAction::kTouchHudModeChange, "TouchHudModeChange"},
    {AcceleratorAction::kTouchFingerprintSensor1, "TouchFingerprintSensor1"},
    {AcceleratorAction::kTouchFingerprintSensor2, "TouchFingerprintSensor2"},
    {AcceleratorAction::kTouchFingerprintSensor3, "TouchFingerprintSensor3"},
    {AcceleratorAction::kUnpin, "Unpin"},
    {AcceleratorAction::kVolumeDown, "VolumeDown"},
    {AcceleratorAction::kVolumeMute, "VolumeMute"},
    {AcceleratorAction::kVolumeUp, "VolumeUp"},
    {AcceleratorAction::kWindowCycleSnapLeft, "WindowCycleSnapLeft"},
    {AcceleratorAction::kWindowCycleSnapRight, "WindowCycleSnapRight"},
    {AcceleratorAction::kWindowMinimize, "WindowMinimize"},
    {AcceleratorAction::kMinimizeTopWindowOnBack, "MinimizeTopWindowOnBack"},
    {AcceleratorAction::kVolumeMuteToggle, "VolumeMuteToggle"},
    {AcceleratorAction::kToggleQuickInsert, "TogglePicker"},
    {AcceleratorAction::kAccessibilityAction, "AccessibilityAction"},
    {AcceleratorAction::kEnableSelectToSpeak, "EnableSelectToSpeak"},
    {AcceleratorAction::kTilingWindowResizeLeft, "TilingWindowResizeLeft"},
    {AcceleratorAction::kTilingWindowResizeRight, "TilingWindowResizeRight"},
    {AcceleratorAction::kTilingWindowResizeUp, "TilingWindowResizeUp"},
    {AcceleratorAction::kTilingWindowResizeDown, "TilingWindowResizeDown"},
    {AcceleratorAction::kToggleMouseKeys, "ToggleMouseKeys"},
    {AcceleratorAction::kResizePipWindow, "ResizePipWindow"},
    {AcceleratorAction::kToggleGeminiApp, "ToggleGeminiApp"},
    {AcceleratorAction::kToggleDoNotDisturb, "ToggleDoNotDisturb"},
    {AcceleratorAction::kToggleCameraAllowed, "ToggleCameraAllowed"},
    {AcceleratorAction::kStartSunfishSession, "StartSunfishSession"},
    {AcceleratorAction::kDebugClearUseKMeansPref, "DebugClearUseKMeansPref"},
    {AcceleratorAction::kDebugKeyboardBacklightToggle,
     "DebugKeyboardBacklightToggle"},
    {AcceleratorAction::kDebugMicrophoneMuteToggle,
     "DebugMicrophoneMuteToggle"},
    {AcceleratorAction::kDebugPrintLayerHierarchy, "DebugPrintLayerHierarchy"},
    {AcceleratorAction::kDebugPrintViewHierarchy, "DebugPrintViewHierarchy"},
    {AcceleratorAction::kDebugPrintWindowHierarchy,
     "DebugPrintWindowHierarchy"},
    {AcceleratorAction::kDebugShowInformedRestore, "DebugShowInformedRestore"},
    {AcceleratorAction::kDebugShowToast, "DebugShowToast"},
    {AcceleratorAction::kDebugShowSystemNudge, "DebugShowSystemNudge"},
    {AcceleratorAction::kDebugSystemUiStyleViewer, "DebugSystemUiStyleViewer"},
    {AcceleratorAction::kDebugToggleDarkMode, "DebugToggleDarkMode"},
    {AcceleratorAction::kDebugToggleDynamicColor, "DebugToggleDynamicColor"},
    {AcceleratorAction::kDebugTogglePowerButtonMenu,
     "DebugTogglePowerButtonMenu"},
    {AcceleratorAction::kDebugToggleShowDebugBorders,
     "DebugToggleShowDebugBorders"},
    {AcceleratorAction::kDebugToggleShowFpsCounter,
     "DebugToggleShowFpsCounter"},
    {AcceleratorAction::kDebugToggleShowPaintRects,
     "DebugToggleShowPaintRects"},
    {AcceleratorAction::kDebugToggleTouchPad, "DebugToggleTouchPad"},
    {AcceleratorAction::kDebugToggleTouchScreen, "DebugToggleTouchScreen"},
    {AcceleratorAction::kDebugToggleTabletMode, "DebugToggleTabletMode"},
    {AcceleratorAction::kDebugToggleVideoConferenceCameraTrayIcon,
     "DebugToggleVideoConferenceCameraTrayIcon"},
    {AcceleratorAction::kDebugToggleWallpaperMode, "DebugToggleWallpaperMode"},
    {AcceleratorAction::kDebugTriggerCrash, "DebugTriggerCrash"},
    {AcceleratorAction::kDebugToggleHudDisplay, "DebugToggleHudDisplay"},
    {AcceleratorAction::kDebugToggleVirtualTrackpad,
     "DebugToggleVirtualTrackpad"},
    {AcceleratorAction::kDevAddRemoveDisplay, "DevAddRemoveDisplay"},
    {AcceleratorAction::kDevToggleAppList, "DevToggleAppList"},
    {AcceleratorAction::kDevToggleUnifiedDesktop, "DevToggleUnifiedDesktop"},
    {AcceleratorAction::kDebugToggleFocusModeState,
     "DebugToggleFocusModeState"},
    {AcceleratorAction::kDebugStartSunfishSession, "DebugStartSunfishSession"},
    {AcceleratorAction::kDebugShowTestWindow, "DebugShowTestWindow"},
});

const char* GetAcceleratorActionName(AcceleratorAction action) {
  // Define the mapping between an AcceleratorAction and its string name.
  // Example:
  //   AcceleratorAction::kDevToggleUnifiedDesktop -> "DevToggleUnifiedDesktop".
  auto iter = kAcceleratorActionToName.find(action);
  CHECK(iter != kAcceleratorActionToName.end());
  return iter->second;
}

base::flat_set<AcceleratorAction> GetAcceleratorActionsForTest() {
  base::flat_set<AcceleratorAction> all;
  for (auto& entry : kAcceleratorActionToName) {
    all.emplace(entry.first);
  }
  return all;
}

}  // namespace ash
